/*-------------------------------------------------------------------------
 *
 * LightDB统一SQL转换库接口头文件
 * 
 *
 * Portions Copyright (c) 2024,  Hundsun Technologies, Inc.
 *
 *-------------------------------------------------------------------------
 */

#ifndef __UNSIQL_H__
#define __UNSIQL_H__

#ifdef __cplusplus
extern "C" {
#endif

#include <stdio.h>
#include <stdlib.h>


typedef signed char GoInt8;
typedef unsigned char GoUint8;
typedef short GoInt16;
typedef unsigned short GoUint16;
typedef int GoInt32;
typedef unsigned int GoUint32;
typedef long long GoInt64;
typedef unsigned long long GoUint64;
typedef GoInt64 GoInt;
typedef GoUint64 GoUint;
typedef size_t GoUintptr;
typedef float GoFloat32;
typedef double GoFloat64;
typedef struct { void *data; GoInt len; GoInt cap; } GoSlice;

#ifdef __GNUC__
    #define UNISQL_DEPRECATED(s)  __attribute__((deprecated))
#else
    #define UNISQL_DEPRECATED(s)
#endif

/**
 * 初始化统一SQL转换库
 * @param options 初始化选项，格式为key=value;key=value字符串
 * 按照技术指导设置初始化选项，不指定option可传空字符串，
 * @return 返回初始化结果，
 *  返回空字符串表示成功，
 *   返回非空字符串表示失败(字符串为错误信息)
 *  调用完成后，注意 调用UnisqlFree 释放返回值的内存
 */
extern char* UnisqlInit(const char *options);

/**
 * 释放内存
 * 注意: 此函数仅能释放统一SQL内部分配的内存，应用自己分配的内存请自行使用配套函数释放
 */
extern void UnisqlFree(void* c) ;
/**
 * 设置还原参数
 * @param n 选项集合的整数表示
 */
extern void UnisqlSetRestoreFormat(long int n);
/**
 * 设置全局参数，参数格式为json字符串
 * @param globalStaticParameterJson json格式的输入参数，包含可设置的所有选项
 */
extern void UnisqlSetGlobalStaticParameter(const char* globalStaticParameterJson);
/**
 * 转换单条SQL语句
 * @param csql 源SQL语句
 * @param sourceDialect 源数据库方言
 * @param targetDialect 目标数据库方言
 * @return 返回转换后的SQL语句，使用完毕后需要调用UnisqlFree函数释放
 */
extern char* UnisqlTransform(const char* csql, const char* sourceDialect, const char* targetDialect);
/**
 * 转换单条SQL语句
 * @param jsonParameter json格式的输入参数，包含源SQL、源数据库方言、源数据库方言即其他选项参数
 * @return 返回转换后的SQL语句，使用完毕后需要调用UnisqlFree函数释放
 */
extern char* UnisqlTransformPro(const char* jsonParameter);

# if defined(_WIN32) || defined(__WIN32__) || defined(__CYGWIN__)
// TransferSQL函数仅支持Linux平台，Windows版本不支持
#else 

// add by zjhua 20250616, pb oracle2oracle19c使用。仅为了兼容pb已经在用。其他用户使用UnisqlTransformSQL。
/**
 * 转换单条SQL语句
 * 注意: 本接口仅支持Linux平台，Windows版本不支持
 * @param sourceDbTypeCode 源数据库方言编码(见UNISQL_DBTYPE_定义)
 * @param targetDbTypeCode 目标数据库方言编码(见UNISQL_DBTYPE_定义)
 * @param sourceSqlStr 源SQL语句
 * @param targetSqlStr 转换后的SQL语句
 * @param targetSqlStrLen 转换后的SQL长度
 * @param jsonParameter json格式的选项
 */
 extern int TransferSQL(
    int sourceDbTypeCode, 
    int targetDbTypeCode, 
    const char* sourceSqlStr, 
    char* targetSqlStr, 
    int targetSqlStrLen, 
    const char* jsonParameter);
#endif
/**
 * 转换单条SQL语句
 * @param sourceDbTypeCode 源数据库方言编码(见UNISQL_DBTYPE_定义)
 * @param targetDbTypeCode 目标数据库方言编码(见UNISQL_DBTYPE_定义)
 * @param sourceSqlStr 源SQL语句
 * @param targetSqlStr 转换后的SQL语句
 * @param targetSqlStrLen 转换后的SQL长度
 * @param jsonParameter json格式的选项
 */
extern int UnisqlTransformSQL(
        int sourceDbTypeCode, 
        int targetDbTypeCode, 
        const char* sourceSqlStr, 
        char* targetSqlStr, 
        int targetSqlStrLen, 
        const char* jsonParameter);
/**
 * 设置垃圾收集目标百分比
 * @param n 百分比值
 */
extern void UnisqlSetGoGc(long int n);
/**
 * 设置垃圾收集的软内存限制
 * @param n 内存值
 */
extern void UnisqlSetGoMemLimit(long int n);
/**
 * 获取insert、update、delete的前后查询SQL，用于多发比对
 * @param jsonParameter json格式的参数集合
 * @return 返回转换后的SQL语句，使用完毕后需要调用UnisqlFree函数释放内存
 */
extern char* UnisqlGetDqlByDml(const char* jsonParameter);
/**
 * 校验go模块版本号是否和输入的版本号version一致，一致则返回00000，否则返回错误提示
 * @param 预期版本号
 * @return 返回错误码，成功返回"00000"，使用完毕后需要调用UnisqlFree函数释放内存
 */
extern char* UnisqlCheckIterationVersion(const char* version);
/**
 * 设置多发比对的表元数据，用于忽略查询字段
 * @param jsonParameter json格式的参数集合
 * @return 使用完毕后需要调用UnisqlFree函数释放内存
 */
extern char* UnisqlInitTableMetas(const char* jsonParameter);
/**
 * 设置表列元数据，用于转换过程中的特殊处理，如改写 replace into
 * @param jsonParameter json格式的参数集合
 * @return 使用完毕后需要调用UnisqlFree函数释放内存
 */
extern char* UnisqlInitTableColumnMetas(const char* jsonParameter);

/**
 * 设置参数
 * 参考: https://light-pg.com/docs/LTSQL/current/configuration/configuration.html#id3
 */
extern void UnisqlSetParamInt(char* paramName, int paramValue);
extern void UnisqlSetParamStr(char* paramName, char* paramValue);
extern void UnisqlSetParamBool(char* paramName, GoUint8 paramValue);

/**
 * 设置会话级别参数，sessionId是通过UnisqlCreateSession函数创建的新会话
 */
extern void UnisqlSetSessionParamStr(char* paramName, char* paramValue, GoUint64 sessionId);

extern char* UnisqlParamStr(char *name, GoUint64 sessionId);

extern char* UnisqlGetTableName(char* encodedParam);

/**
 * 解析简单SQL语句
 * @param sourceSql 待解析的SQL语句
 * @return 解析结果，使用完毕后需要调用UnisqlFree函数释放内存
 */
extern char* UnisqlParseSimpleStatement(const char* sourceSql);

/**
 * 解析简单SQL语句
 * @param sourceSql 待解析的SQL语句
 * @return 解析结果，使用完毕后需要调用UnisqlFree函数释放内存
 */
 extern char* UnisqlParseSimpleStatementInfo(const char* sourceSql);



/**
 * 新建一个session实例
 * 返回session对象句柄
 * 注意: 使用完毕后务必调用UnisqlFreeSession进行释放
 */
extern GoUint64 UnisqlCreateSession();

// 关闭session
//
extern void UnisqlFreeSession(GoUint64 sessionId);


/**
 * 获取版本号
 * @return 版本号，使用完毕后需要调用UnisqlFree函数释放内存
 */
extern char* UnisqlVersion();



/**
 * 以下函数废弃
 */
UNISQL_DEPRECATED("本函数已经废弃，请使用 UnisqlFree 函数")
extern void FreePointer(void* c);

UNISQL_DEPRECATED("本函数已经废弃，请使用 UnisqlSetRestoreFormat 函数")
extern void SetRestoreFormat(long int n);

UNISQL_DEPRECATED("本函数已经废弃，请使用 UnisqlSetGlobalStaticParameter 函数")
extern void SetGlobalStaticParameter(char* globalStaticParameterJson);

UNISQL_DEPRECATED("本函数已经废弃，请使用 UnisqlTransform 函数")
extern char* Transform(char* csql, char* sourceDialect, char* targetDialect);

UNISQL_DEPRECATED("本函数已经废弃，请使用 UnisqlSetParamInt 函数")
extern void SetParamInt(char* paramName, int paramValue);

UNISQL_DEPRECATED("本函数已经废弃，请使用 UnisqlSetParamStr 函数")
extern void SetParamStr(char* paramName, char* paramValue);

UNISQL_DEPRECATED("本函数已经废弃，请使用 UnisqlSetParamBool 函数")
extern void SetParamBool(char* paramName, GoUint8 paramValue);

UNISQL_DEPRECATED("本函数已经废弃，请使用 UnisqlTransformPro 函数")
extern char* TransformPro(char* encodedParam);

UNISQL_DEPRECATED("本函数已经废弃，请使用 UnisqlTransformSQL 函数")
extern int TransformSQL(int sourceDbTypeCode, int targetDbTypeCode, char* sourceSqlStr, char* targetSqlStr, int targetSqlMemoryLen, char* extendParam);

UNISQL_DEPRECATED("本函数已经废弃，请使用 UnisqlSetGoGc 函数")
extern void SetGoGc(long int n);

UNISQL_DEPRECATED("本函数已经废弃，请使用 UnisqlSetGoMemLimit 函数")
extern void SetGoMemLimit(long int n);

UNISQL_DEPRECATED("本函数已经废弃，请使用 UnisqlGetDqlByDml 函数")
extern char* GetDqlByDml(char* encodedParam);

UNISQL_DEPRECATED("本函数已经废弃，请使用 UnisqlCheckIterationVersion 函数")
extern char* CheckIterationVersion(char* version);

UNISQL_DEPRECATED("本函数已经废弃，请使用 UnisqlInitTableMetas 函数")
extern char* InitTableMetas(char* jsonParameter);

UNISQL_DEPRECATED("本函数已经废弃，请使用 UnisqlInitTableColumnMetas 函数")
extern char* InitTableColumnMetas(char* jsonParameter);

UNISQL_DEPRECATED("本函数已经废弃，请使用 UnisqlGetTableName 函数")
extern char* GetTableName(char* encodedParam);

UNISQL_DEPRECATED("本函数已经废弃，请使用 UnisqlCheckCompatibleScanSql 函数")
extern char* CheckCompatibleScanSql(char* encodedParam);



#ifdef __cplusplus
}
#endif

#endif 
