-- SQL --  创建unisql 数据库 25.3.2.2版本脚本
CREATE DATABASE IF NOT EXISTS unisql default charset utf8mb4;

-- SQL --  删除GrantUnisqlPermissions
DROP PROCEDURE IF EXISTS unisql.GrantUnisqlPermissions;

-- SQL --  创建GrantUnisqlPermissions
CREATE PROCEDURE unisql.GrantUnisqlPermissions()
BEGIN
  DECLARE done INT DEFAULT 0;
  DECLARE username VARCHAR(100);
  DECLARE cur CURSOR FOR SELECT concat('\'',User, '\'','@','\'',Host,'\'')  FROM mysql.user;
DECLARE CONTINUE HANDLER FOR NOT FOUND SET done = 1;

OPEN cur;
read_loop: LOOP
    FETCH cur INTO username;
    IF done THEN
      LEAVE read_loop;
END IF;

    SET @grant_stmt = CONCAT('GRANT EXECUTE, SELECT, UPDATE, INSERT ON unisql.* TO ', username, ' with grant option;');
PREPARE stmt FROM @grant_stmt;
EXECUTE stmt;
DEALLOCATE PREPARE stmt;
END LOOP;

CLOSE cur;
END ;

-- SQL --  执行GrantUnisqlPermissions
call unisql.GrantUnisqlPermissions();

-- SQL --  创建ltrim的自定义函数

-- 为什么varchar类型定义为varchar(16383) ?
--  16383是goldendb的varchar类型最大长度(字符)， 为了保持最大兼容性，直接使用最大长度
--  在LDP的dbclient的C代码中有获取返回类型长度用于分配字符串内存,如果修改需要注意核实.
CREATE OR REPLACE FUNCTION unisql.ltrim(p_str VARCHAR(16383), p_chars VARCHAR(16383))
RETURNS VARCHAR(16383)
DETERMINISTIC
READS SQL DATA
BEGIN
    DECLARE v_ret VARCHAR(16383);
    IF p_str IS NULL THEN
        RETURN NULL;
    END IF;
    -- 在mysql中''和' '等价的，在oracle中不等，如果需要模拟oracle的'',使用length(p_chars)=0
    IF p_chars IS null or CHAR_LENGTH(p_chars)=0 THEN
        RETURN NULL;
    END IF;

    SET p_chars = REPLACE(p_chars, '\\', '\\\\');
    SET p_chars = REPLACE(p_chars, ']', '\\]');
    SET p_chars = REPLACE(p_chars, '[', '\\[');
    SET p_chars = REPLACE(p_chars, '-', '\\-');
    SET p_chars = REPLACE(p_chars, '^', '\\^');

    SET v_ret = REGEXP_REPLACE(p_str, CONCAT('^[', p_chars, ']+'), '');

    IF CHAR_LENGTH(v_ret) = 0 THEN
        RETURN NULL;
    END IF;
    RETURN v_ret;
END;

-- SQL --  创建rtrim的自定义函数

-- 为什么varchar类型定义为varchar(16383) ? 参考ltrim说明
CREATE OR REPLACE FUNCTION unisql.rtrim(p_str VARCHAR(16383), p_chars VARCHAR(16383))
RETURNS VARCHAR(16383)
DETERMINISTIC
READS SQL DATA
BEGIN
    DECLARE v_ret VARCHAR(16383);
    IF p_str IS NULL THEN
        RETURN NULL;
    END IF;
    -- 在mysql中''和' '等价的，在oracle中不等，如果需要模拟oracle的'',使用length(p_chars)=0
    IF p_chars IS null or CHAR_LENGTH(p_chars)=0 THEN
        RETURN NULL;
    END IF;

    SET p_chars = REPLACE(p_chars, '\\', '\\\\');
    SET p_chars = REPLACE(p_chars, ']', '\\]');
    SET p_chars = REPLACE(p_chars, '[', '\\[');
    SET p_chars = REPLACE(p_chars, '-', '\\-');
    SET p_chars = REPLACE(p_chars, '^', '\\^');

    SET v_ret = REGEXP_REPLACE(p_str, CONCAT('[', p_chars, ']+$'), '');

    IF CHAR_LENGTH(v_ret) = 0 THEN
        RETURN NULL;
    END IF;
    RETURN v_ret;
END;

-- SQL --  创建 all_ind_columns 的自定义视图
CREATE OR REPLACE VIEW unisql.all_ind_columns
AS
SELECT
    UPPER(s.TABLE_SCHEMA) AS INDEX_OWNER,
    UPPER(s.INDEX_NAME) AS INDEX_NAME,
    UPPER(s.TABLE_SCHEMA) AS TABLE_OWNER,
    UPPER(s.TABLE_NAME) AS TABLE_NAME,
    UPPER(s.COLUMN_NAME) AS COLUMN_NAME,
    s.SEQ_IN_INDEX AS COLUMN_POSITION,
    CASE
        WHEN c.CHARACTER_MAXIMUM_LENGTH IS NOT NULL
            THEN c.CHARACTER_MAXIMUM_LENGTH
        WHEN c.NUMERIC_PRECISION IS NOT NULL
            THEN c.NUMERIC_PRECISION
        WHEN c.DATETIME_PRECISION IS NOT NULL AND c.DATETIME_PRECISION != 0
            THEN c.DATETIME_PRECISION
        ELSE NULL
    END AS COLUMN_LENGTH
FROM
    information_schema.STATISTICS s
JOIN
    information_schema.COLUMNS c
    ON s.TABLE_SCHEMA = c.TABLE_SCHEMA
    AND s.TABLE_NAME = c.TABLE_NAME
    AND s.COLUMN_NAME = c.COLUMN_NAME
ORDER BY
    s.TABLE_SCHEMA,
    s.TABLE_NAME,
    s.INDEX_NAME,
    s.SEQ_IN_INDEX;

-- SQL --  创建 user_ind_columns 的自定义视图
CREATE OR REPLACE VIEW unisql.user_ind_columns
AS
SELECT
    INDEX_NAME,
    TABLE_NAME,
    COLUMN_NAME,
    COLUMN_POSITION,
    COLUMN_LENGTH
FROM
    unisql.all_ind_columns c
WHERE
    c.TABLE_OWNER = DATABASE();


-- SQL --  模拟视图 all_users
CREATE OR REPLACE VIEW unisql.all_users AS
SELECT
    upper(u.user) AS `USERNAME`,
    CAST(NULL AS DECIMAL(38,0)) AS `USER_ID`,
    password_last_changed       AS `CREATED`
FROM mysql.user u;

-- SQL --  模拟视图 user_objects
CREATE OR REPLACE VIEW unisql.user_objects AS
SELECT
    -- 每个列尾部都as 反引号大写的名称是为了统一，保持和Oracle行为一样列名大写
    upper(a.object_name) COLLATE utf8_general_ci as `OBJECT_NAME`,
    a.subobject_name COLLATE utf8_general_ci as `SUBOBJECT_NAME`,
    a.object_id as `OBJECT_ID`,
    a.data_object_id as `DATA_OBJECT_ID`,
    -- 命中 sequence 时，object_type 强制为 SEQUENCE,在目标库中的ALL_OBJECT中SEQUENCE 被视为TABLE
    CASE
        WHEN s.sequence_name IS NOT null and a.object_type = 'TABLE' THEN 'SEQUENCE'
        ELSE a.object_type
        END AS `OBJECT_TYPE`,
    a.created as `CREATED`,
    a.last_ddl_time as `LAST_DDL_TIME` ,
    a.timestamp as `TIMESTAMP`,
    a.status as `STATUS`
FROM information_schema.ALL_OBJECTS a
         LEFT JOIN information_schema.ALL_SEQUENCES s
                   ON s.sequence_owner = DATABASE()
                       AND s.sequence_name  = a.object_name
WHERE a.owner = DATABASE()
  -- 仅仅保证table(sequence)\view\index
  and a.object_type in('TABLE','VIEW','INDEX');

-- SQL --  模拟视图 all_objects
CREATE OR REPLACE VIEW unisql.all_objects AS
SELECT
    -- 每个列尾部都as 反引号大写的名称是为了统一，保持和Oracle行为一样列名大写
    upper(a.owner) COLLATE utf8_general_ci as `OWNER`,
    upper(a.object_name) COLLATE utf8_general_ci as `OBJECT_NAME`,
    a.subobject_name as `SUBOBJECT_NAME`,
    a.object_id as `OBJECT_ID`,
    a.data_object_id as `DATA_OBJECT_ID`,
    -- 命中 sequence 时，object_type 强制为 SEQUENCE,在目标库中的ALL_OBJECT中SEQUENCE 被视为TABLE
    CASE
        WHEN s.sequence_name IS NOT null and a.object_type = 'TABLE' THEN 'SEQUENCE'
        ELSE a.object_type
        END AS `OBJECT_TYPE`,
    a.created as `CREATED`,
    a.last_ddl_time as `LAST_DDL_TIME`,
    a.timestamp as `TIMESTAMP`,
    a.status as `STATUS`
FROM information_schema.ALL_OBJECTS a
         LEFT JOIN information_schema.ALL_SEQUENCES s
                   ON  s.sequence_name  = a.object_name
-- 仅仅保证table(sequence)\view\index
where a.object_type in('TABLE','VIEW','INDEX');



-- SQL --  创建用于支持oracle的all_indexes视图
create or replace
view unisql.all_indexes
as
SELECT
    upper(t.OWNER) as OWNER,
    upper(t.INDEX_NAME) as INDEX_NAME,
    upper(cast(t.INDEX_TYPE as varchar(10))) as INDEX_TYPE,
    upper(t.TABLE_OWNER) as TABLE_OWNER,
    upper(t.TABLE_NAME) as TABLE_NAME,
    t.TABLE_TYPE,
    t.UNIQUENESS
FROM (
         -- 子查询：按索引唯一标识分组，取第一条数据
         SELECT
             *,
             ROW_NUMBER() OVER (
      PARTITION BY OWNER, TABLE_NAME, INDEX_NAME  -- 索引唯一标识
      ORDER BY INDEX_COLUMN_NAME  -- 排序无意义，仅取第一条
    ) AS rn
         FROM information_schema.all_indexes
     ) t
WHERE t.rn = 1;



-- SQL --  创建用于支持oracle的user_indexes视图
create or replace
view unisql.user_indexes
as
select
    t.INDEX_NAME,
    t.INDEX_TYPE,
    t.TABLE_OWNER,
    t.TABLE_NAME,
    t.TABLE_TYPE,
    t.UNIQUENESS
from unisql.all_indexes t
WHERE t.owner = DATABASE();





-- SQL --  创建用于支持oracle的user_tables视图
create or replace
view unisql.user_tables
as
select
    upper(t.TABLE_NAME) COLLATE utf8_general_ci as TABLE_NAME ,
    t.TABLESPACE_NAME ,
    upper(cast(t.CLUSTER_NAME as varchar(4))) COLLATE utf8mb4_general_ci as CLUSTER_NAME ,
    upper(cast(t.IOT_NAME as varchar(4))) COLLATE utf8mb4_general_ci as IOT_NAME ,
    t.STATUS
from
    information_schema.all_tables t
WHERE t.owner = DATABASE();






-- SQL --  创建用于支持oracle的all_tab_columns视图
create or replace
view unisql.all_tab_columns
as
select
    upper(t.OWNER) as OWNER ,
    upper(t.TABLE_NAME) as TABLE_NAME ,
    upper(t.COLUMN_NAME) as COLUMN_NAME ,
    upper(cast(c.DATA_TYPE as VARCHAR(64))) as DATA_TYPE,
    upper(cast(t.DATA_TYPE_MOD as varchar(3))) as DATA_TYPE_MOD,
    upper(cast(t.DATA_TYPE_OWNER as varchar(128))) as DATA_TYPE_OWNER,
    t.DATA_LENGTH ,
    t.DATA_PRECISION
from
    information_schema.all_tab_cols t
        left join information_schema.COLUMNS c
                  on
                      t.OWNER = c.TABLE_SCHEMA
                          and t.TABLE_NAME = c.TABLE_NAME
                          and t.COLUMN_NAME = c.COLUMN_NAME;



-- SQL --  创建用于支持oracle的user_tab_columns视图
create or replace
view unisql.user_tab_columns
as
select
    t.TABLE_NAME ,
    t.COLUMN_NAME ,
    t.DATA_TYPE ,
    t.DATA_TYPE_MOD ,
    t.DATA_TYPE_OWNER ,
    t.DATA_LENGTH ,
    t.DATA_PRECISION
from
    unisql.all_tab_columns t
WHERE t.owner = DATABASE();

-- SQL --  创建substr的自定义函数
CREATE OR REPLACE FUNCTION unisql.substr2(p_str VARCHAR(16383), p_pos INT)
RETURNS VARCHAR(16383)
DETERMINISTIC
READS SQL DATA
BEGIN
    IF p_pos = 0 THEN
        SET p_pos = 1;
    END IF;
    RETURN SUBSTR(p_str, p_pos);
END;

-- SQL --  创建substr的自定义函数
CREATE OR REPLACE FUNCTION unisql.substr3(p_str VARCHAR(16383), p_pos INT,p_len INT)
RETURNS VARCHAR(16383)
DETERMINISTIC
READS SQL DATA
BEGIN
    IF p_pos = 0 THEN
        SET p_pos = 1;
    END IF;
    RETURN SUBSTR(p_str, p_pos, p_len);
END;

-- SQL --  创建lpad的自定义函数
CREATE or replace FUNCTION unisql.lpad(str VARCHAR(16383) CHARSET utf8mb4,display_width numeric(38,30),pad_str VARCHAR(16383) CHARSET utf8mb4)
RETURNS VARCHAR(16383) CHARSET utf8mb4
DETERMINISTIC
READS SQL DATA
SQL SECURITY INVOKER
BEGIN
    DECLARE result VARCHAR(16383) DEFAULT '';
    DECLARE padded VARCHAR(16383) DEFAULT '';
    DECLARE i, j INT DEFAULT 1;
    DECLARE total_width, char_width, pad_width, needed_width INT DEFAULT 0;
    DECLARE str_len, pad_len INT DEFAULT 0;
    -- 处理边界情况
    IF display_width is null or display_width <= 0 THEN
        RETURN NULL;
    END IF;
    -- 对于浮点数向0取整
	set display_width = TRUNCATE(display_width, 0);
	-- null和空字符串都返回null
    IF str IS NULL OR str = '' THEN
        RETURN NULL;
    END IF;
	-- null和空字符串都返回null
    IF pad_str IS NULL OR pad_str = '' THEN
        RETURN NULL;
    END IF;
    SET str_len = CHAR_LENGTH(str);
    SET pad_len = CHAR_LENGTH(pad_str);
    -- 第一步：计算输入字符串的实际显示宽度，并截断到不超过 display_width
    SET total_width = 0;
    SET result = '';
    compute_str_width: WHILE i <= str_len DO
        -- 需要使用 SUBSTRING(str, i, 1) 对于空格无法 set赋值;
        IF LENGTH(SUBSTRING(str, i, 1)) = 1 AND ASCII(SUBSTRING(str, i, 1)) BETWEEN 32 AND 126 THEN
            SET char_width = 1;
        ELSE
            SET char_width = 2;
        END IF;

        IF total_width + char_width > display_width THEN
            LEAVE compute_str_width;
        END IF;

        SET result = CONCAT(result, SUBSTRING(str, i, 1));
        SET total_width = total_width + char_width;
        SET i = i + 1;
    END WHILE compute_str_width;
    -- 如果已经满足长度，直接返回
    IF total_width >= display_width THEN
        RETURN result;
    END IF;
    -- 第二步：计算还需要多少显示宽度
    SET needed_width = display_width - total_width;
    -- 第三步：用 pad_str 左填充（注意：填充也是按显示宽度！）
    build_padding: WHILE CHAR_LENGTH(padded) < 10000 AND needed_width > 0 DO  -- 防止无限循环
        SET j = 1;
        pad_loop: WHILE j <= pad_len AND needed_width > 0 DO
            -- 需要使用 SUBSTRING(str, i, 1) 对于空格无法 set赋值;
            IF LENGTH(SUBSTRING(pad_str, j, 1)) = 1 AND ASCII(SUBSTRING(pad_str, j, 1)) BETWEEN 32 AND 126 THEN
                SET pad_width = 1;
            ELSE
                SET pad_width = 2;
            END IF;
            IF needed_width >= pad_width THEN
                SET padded = CONCAT(padded, SUBSTRING(pad_str, j, 1));
                SET needed_width = needed_width - pad_width;
            ELSE
                -- 剩余宽度不够放一个 pad 字符，跳过（Oracle 也会跳过）
                SET needed_width = 0;
            END IF;
            SET j = j + 1;
        END WHILE pad_loop;
    END WHILE build_padding;
    -- 第四步：拼接（填充在左，原字符串在右）
    RETURN CONCAT(padded, result);
END;

-- SQL --  创建rpad的自定义函数
CREATE OR REPLACE FUNCTION unisql.rpad(
    str VARCHAR(16383) CHARSET utf8mb4,
    display_width NUMERIC(38,30),
    pad_str VARCHAR(16383) CHARSET utf8mb4
)
RETURNS VARCHAR(16383) CHARSET utf8mb4
DETERMINISTIC
READS SQL DATA
SQL SECURITY INVOKER
BEGIN
    DECLARE result VARCHAR(16383) DEFAULT '';
    DECLARE padded VARCHAR(16383) DEFAULT '';
    DECLARE i, j INT DEFAULT 1;
    DECLARE total_width, char_width, pad_width, needed_width INT DEFAULT 0;
    DECLARE str_len, pad_len INT DEFAULT 0;

    -- 处理边界情况
    IF display_width IS NULL OR display_width <= 0 THEN
        RETURN NULL;
    END IF;

    -- 对浮点数向零取整
    SET display_width = TRUNCATE(display_width, 0);

    -- null 和空字符串都返回 null
    IF str IS NULL OR str = '' THEN
        RETURN NULL;
    END IF;

    IF pad_str IS NULL OR pad_str = '' THEN
        RETURN NULL;
    END IF;

    SET str_len = CHAR_LENGTH(str);
    SET pad_len = CHAR_LENGTH(pad_str);

    -- 第一步：计算输入字符串的实际显示宽度，并截断到不超过 display_width
    SET total_width = 0;
    SET result = '';
    SET i = 1;  -- 显式重置 i（安全起见）

    compute_str_width: WHILE i <= str_len DO
        IF LENGTH(SUBSTRING(str, i, 1)) = 1 AND ASCII(SUBSTRING(str, i, 1)) BETWEEN 32 AND 126 THEN
            SET char_width = 1;
        ELSE
            SET char_width = 2;
        END IF;

        IF total_width + char_width > display_width THEN
            LEAVE compute_str_width;
        END IF;

        SET result = CONCAT(result, SUBSTRING(str, i, 1));
        SET total_width = total_width + char_width;
        SET i = i + 1;
    END WHILE compute_str_width;

    -- 如果已经满足长度，直接返回
    IF total_width >= display_width THEN
        RETURN result;
    END IF;

    -- 第二步：计算还需要多少显示宽度
    SET needed_width = display_width - total_width;

    -- 第三步：用 pad_str 右填充（按显示宽度逐字符）
    SET padded = '';
    build_padding: WHILE CHAR_LENGTH(padded) < 10000 AND needed_width > 0 DO
        SET j = 1;
        pad_loop: WHILE j <= pad_len AND needed_width > 0 DO
            IF LENGTH(SUBSTRING(pad_str, j, 1)) = 1 AND ASCII(SUBSTRING(pad_str, j, 1)) BETWEEN 32 AND 126 THEN
                SET pad_width = 1;
            ELSE
                SET pad_width = 2;
            END IF;

            IF needed_width >= pad_width THEN
                SET padded = CONCAT(padded, SUBSTRING(pad_str, j, 1));
                SET needed_width = needed_width - pad_width;
            ELSE
                SET needed_width = 0;
            END IF;

            SET j = j + 1;
        END WHILE pad_loop;
    END WHILE build_padding;
    -- 第四步：拼接（原字符串在左，填充在右）
    RETURN CONCAT(result, padded);
END;


